#ifndef __RPGGAME_H__
#define __RPGGAME_H__

#include "cube.h"

#define SPAWNNAMELEN 64

// Constants (placed here for easy access)
#define DEFAULTMODEL "rc"
#define DEFAULTMAP "title"

enum                            // static entity types
{
	NOTUSED = ET_EMPTY,         // entity slot not in use in map
	LIGHT = ET_LIGHT,           // lightsource, attr1 = radius, attr2 = intensity
	MAPMODEL = ET_MAPMODEL,     // attr1 = angle, attr2 = idx
	PLAYERSTART,                // attr1 = angle, attr2 = team
	ENVMAP = ET_ENVMAP,         // attr1 = radius
	PARTICLES = ET_PARTICLES,
	MAPSOUND = ET_SOUND,
	SPOTLIGHT = ET_SPOTLIGHT,
	SPAWN = ET_GAMESPECIFIC,
	TELEDEST, //attr1 = yaw, attr2 = from
	JUMPPAD, //attr1 = Z, attr2 = Y, attr3 = X, attr4 = radius
	CHECKPOINT,
	MAXENTTYPES
};

struct rpgentity : extentity
{
    char name[SPAWNNAMELEN];

    rpgentity() { memset(name, 0, SPAWNNAMELEN); }
};

struct projeffect
{
	//these respond to the particle's trail/projectile
	//TODO sounds
	const char *mdl;
	int basevel, kickback;

	bool trail;
	int trailpart, trailcol, trailfade, gravity; //note for gravity, lower is faster
	float trailsize;

	int projpart, projcol;
	float projsize;

	bool dynlight;
	int lightradius, lightcolour;

	int deathpart, deathpartcol, deathfade, deathlightinitcol, deathlightflags, deathdecal, deathlightfade;
	float deathpartsize;

	projeffect()
	{
		mdl = NULL;
		basevel = 200;
		kickback = 50;

		trail = true;
		trailpart = PART_STEAM;
		trailcol = 0x7F7F7F;
		trailsize = 1.0f;
		trailfade = 500;
		gravity = 200;

		projpart = PART_FIREBALL1;
		projcol = 0xFFBF00;
		projsize = 4;

		dynlight = true;
		lightradius = 64;
		lightcolour = 0xFFBF00;


		deathpart = PART_EXPLOSION;
		deathpartcol = 0xFFBF00;
		deathpartsize = 4.0f;
		deathfade = 800;
		deathdecal = 0;
		deathlightinitcol = lightcolour;
		deathlightflags = DL_EXPAND;
		deathlightfade = 200;

		deathdecal = DECAL_BURN;
	}
	~projeffect() {}
};

struct rpgent;

struct projectile
{
	//TODO dedicated soundchannel
	rpgent &weapon; //the item it takes it's effect/damage stack from
	vec o, d;
	int speed, radius;
	int effect, gravity; //ignore projpart, projcol and projsize if magic is false
	entitylight light;

	projectile(rpgent &ent) : weapon(ent)
	{
		o = d = vec(0, 0, 0);
		speed = 100;
		radius = 32;
		effect = 0, gravity = 0;
	}
	~projectile() {}
};

enum //status effects, both negative and positive
{
	STATUS_HEALTH = 0, //attributes
	STATUS_MOVE,
	STATUS_STRENGTH,
	STATUS_INTELLIGENCE,
	STATUS_CHARISMA,
	STATUS_ENDURANCE,
	STATUS_AGILITY,
	STATUS_LUCK,
	STATUS_CRIT,
	STATUS_HREGEN,
	STATUS_MREGEN,
	STATUS_FIRE, //these are resistances
	STATUS_WATER,
	STATUS_AIR,
	STATUS_EARTH,
	STATUS_MAGIC,
	STATUS_SLASH,
	STATUS_BLUNT,
	STATUS_PIERCE,
	STATUS_DISPELL, //negative strength dispells positive magics, positive strength bad magics
	STATUS_DOOM,
	STATUS_REFLECT,
	STATUS_INVIS,
	STATUS_LIGHT,
	STATUS_DEATH, //rather a status for withdrawal symptoms of death, as in, your stats are temporarily lowered due to your recent death
	/*Additional Spells worth considerings
	STATUS_STUN,
	STATUS_POLYMORPH, <-- this dispells all other polymorphs present
	*/
	STATUS_MAX
};



struct status
{
	int type, strength, applied, duration, startmillis, resists; //note applied is mostly for DoT attacks/healings resists is the applicable resistances, see the ATTACK* enum

	status(int t, int s, int d, int at = -1) : type(t), strength(s), duration(d), startmillis(lastmillis), resists(at)
	{
		if(duration <= 0) duration = 1;
		applied = 0;
	}
	~status() {};
};

//note, none-elemental attacks only use the respective magic/damage resistance
//otherwise if the attack is elemental, use a two step algorithm, eg, if both resistances are 25%, take 56.25% damage
//also note that elemental attacks will do 25% extra damage to compensate for having two resistances, so in the above case, 70.3125% damage will be taken
//note that a resistance is maxed at 80% which corresponds to 5% in elental attacks and 20% for regular attacks

enum
{
	RESIST_FIRE = 0,
	RESIST_WATER,
	RESIST_AIR,
	RESIST_EARTH,
	RESIST_MAGIC,
	RESIST_SLASH,
	RESIST_BLUNT,
	RESIST_PIERCE,
	/* Resistances under consideration

	RESIST_POISON

	*/
	RESIST_MAX
};

enum
{
	ATTACK_SLASHING = 0,
	ATTACK_BLUNT = 1,
	ATTACK_PIERCE = 2,
	ATTACK_MAGIC = 3,

	//keep a bit above the above, eg, if the above goes to ATTACK_* = 6, then use 8, the rest will automatically update the numbers
	ATTACK_FIRE = 4,
	ATTACK_WATER = ATTACK_FIRE<<1, //8
	ATTACK_AIR = ATTACK_FIRE|ATTACK_WATER, //12
	ATTACK_EARTH = ATTACK_FIRE<<2, //16

	/* Attack Types under consideration

	ATTACK_POISON = ATTACK_EARTH|ATTACK_FIRE,   //20

	*/
};

enum
{
	CROSS_DEFAULT = 0,
	CROSS_EDIT,
	CROSS_TALK,
	CROSS_PICKUP
};

enum
{
	SKILL_BLADE,
	SKILL_MARKSMAN,
	SKILL_UNARMED,
	SKILL_GUN,
	SKILL_REPAIR,
	SKILL_THROW,
	SKILL_HEAL,
	SKILL_MAGIC,
	SKILL_BARTER,
	SKILL_SNEAK,
	SKILL_MAX
};

struct stats
{
	//stats; max at 100
	int strength, intelligence, charisma, endurance, agility, luck;

	//skills
	struct skill
	{
		int level, exp;
	} skills[SKILL_MAX];

	//attributes
	int resistance[RESIST_MAX], maxhealth, maxmana, maxweight, movespeed, jumpvel;
	float healthregen, manaregen;

	void calcattrs()
	{
		loopi(RESIST_MAX)
		{
			switch(i)
			{
				case RESIST_FIRE:
					resistance[i] = agility/4.0f + charisma / 6.0f + intelligence / 6.0f;
					break;
				case RESIST_WATER:
					resistance[i] = endurance/4.0f + charisma / 6.0f + intelligence / 6.0f;
					break;
				case RESIST_AIR:
					resistance[i] = strength/8.0f  + endurance/8.0f + charisma / 6.0f + intelligence / 6.0f;
					break;
				case RESIST_EARTH:
					resistance[i] = strength/4.0f + charisma / 6.0f + intelligence / 6.0f;
					break;
				case RESIST_MAGIC:
					resistance[i] = charisma / 3.0f + intelligence / 3.0f;
					break;
				case RESIST_SLASH:
					resistance[i] = strength/2;
					break;
				case RESIST_BLUNT:
					resistance[i] = endurance/2;
					break;
				case RESIST_PIERCE:
					resistance[i] = agility/2;
					break;
				default: //ie, unknown resistance
					resistance[i] = 10;
					break;
			}
			resistance[i] += luck/5;
		}
		maxhealth = 10 + endurance * 4 + strength * 2;
		healthregen = endurance / 8.0f + strength / 16.0f;

		maxmana = 10 + intelligence * 4 + charisma * 2;
		manaregen = intelligence / 6.0f + charisma / 12.0f;

		maxweight = strength * 3 + endurance;

		movespeed = 30 + agility * .9f; //ends at 120
		jumpvel =  100 + agility; //ends at 200
	}

	void limits()
	{
		#define limit(n, mn, mx) n = min(mx, max(n, mn))
		#define limitdn(n, mn) n = max(n, mn)
		limit(strength, 5, 100);
		limit(intelligence, 5, 100);
		limit(charisma, 5, 100);
		limit(endurance, 5, 100);
		limit(agility, 5, 100);
		limit(luck, 5, 100);
		limit(movespeed, 20, 150);
		limit(jumpvel, 66, 250);
		loopi(RESIST_MAX) limit(resistance[i], 0, 80);

		limitdn(healthregen, (float) 0.01);
		limitdn(manaregen, (float) 0.01);
		limitdn(maxhealth, 1);
		limitdn(maxmana, 1);
		limitdn(maxweight, 20);
		#undef limit
		#undef limitdn
	}

	stats(bool empty = false)
	{
		if(!empty)
		{
			strength = intelligence = charisma = endurance = agility = luck = 20;
			loopi(SKILL_MAX) {skills[i].level = 10; skills[i].exp = 512000;}
			calcattrs();
		}
		else
		{
			strength = intelligence = charisma = endurance = agility = luck = maxhealth = healthregen = maxmana = manaregen = maxweight = movespeed = jumpvel = 0;
			loopi(SKILL_MAX) {skills[i].level = 0; skills[i].exp = 0;}
			loopi(RESIST_MAX) resistance[i] = 0;
		}
	}
	~stats() {}

	#define test(test) if(strength test s.strength || intelligence test s.intelligence || endurance test s.endurance || \
			agility test s.agility || luck test s.luck || maxhealth test s.maxhealth || healthregen test s.healthregen || \
			maxmana test s.maxmana || manaregen test s.manaregen || maxweight test s.maxweight || movespeed test s.movespeed) return false; \
		loopi(SKILL_MAX) if(skills[i].level test s.skills[i].level) return false; \
		loopi(RESIST_MAX) if (resistance[i] test s.resistance[i]) return false;

	bool operator>= (const stats &s)
	{
		test(<)

		return true;
	}
	bool operator< (const stats &s)
	{
		test(>=)

		return true;
	}
	bool operator== (const stats &s)
	{
		test(!=)

		return true;
	}

	#undef test

	stats &add(int n)
	{
		strength += n, intelligence += n, charisma += n, endurance += n, agility += n, luck += n;
		loopi(SKILL_MAX) skills[i].level += n;
		loopi(RESIST_MAX) resistance[i] += n;
		return *this;
	}
	stats &add(const stats &s)
	{
		strength += s.strength, intelligence += s.intelligence, charisma += s.charisma, endurance += s.endurance, agility += s.agility, luck += s.luck;
		loopi(SKILL_MAX) skills[i].level += s.skills[i].level;
		loopi(RESIST_MAX) resistance[i] += s.resistance[i];
		return *this;
	}
	stats &addall(int n)
	{
		add(n);
		maxhealth += n, healthregen += n, maxmana += n, manaregen += n, maxweight += n, movespeed += n, jumpvel += n;
		return *this;
	}
	stats &addall(const stats &s)
	{
		add(s);
		maxhealth += s.maxhealth, healthregen += s.healthregen, maxmana += s.maxmana, manaregen += s.manaregen, maxweight += s.maxweight, movespeed += s.movespeed, jumpvel += s.jumpvel;
		return *this;
	}
	stats &sub(int n)
	{
		strength -= n, intelligence -= n, charisma -= n, endurance -= n, agility -= n, luck -= n;
		loopi(SKILL_MAX) skills[i].level -= n;
		loopi(RESIST_MAX) resistance[i] -= n;
		return *this;
	}
	stats &sub(const stats &s)
	{
		strength -= s.strength, intelligence -= s.intelligence, charisma -= s.charisma, endurance -= s.endurance, agility -= s.agility, luck -= s.luck;
		loopi(SKILL_MAX) skills[i].level -= s.skills[i].level;
		loopi(RESIST_MAX) resistance[i] -= s.resistance[i];
		return *this;
	}
	stats &suball(int n)
	{
		sub(n);
		maxhealth -= n, healthregen -= n, maxmana -= n, manaregen -= n, maxweight -= n, movespeed -= n, jumpvel -= n;
		return *this;
	}
	stats &suball(const stats &s)
	{
		sub(s);
		maxhealth -= s.maxhealth, healthregen -= s.healthregen, maxmana -= s.maxmana, manaregen -= s.manaregen, maxweight -= s.maxweight, movespeed -= s.movespeed, jumpvel -= s.jumpvel;
		return *this;
	}
	stats &div(float n)
	{
		strength /= n, intelligence /= n, charisma /= n, endurance /= n, agility /= n, luck /= n;
		loopi(SKILL_MAX) skills[i].level /= n;
		loopi(RESIST_MAX) resistance[i] /= n;
		return *this;
	}
	stats &divall(float n)
	{
		div(n);
		maxhealth /= n, healthregen /= n, maxmana /= n, manaregen /= n, maxweight /= n, movespeed /= n, jumpvel /= n;
		return *this;
	}
	stats &mul(float n)
	{
		strength *= n, intelligence *= n, charisma *= n, endurance *= n, agility *= n, luck *= n;
		loopi(SKILL_MAX) skills[i].level *= n;
		loopi(RESIST_MAX) resistance[i] *= n;
		return *this;
	}
	stats &mulall(float n)
	{
		mul(n);
		maxhealth *= n, healthregen *= n, maxmana *= n, manaregen *= n, maxweight *= n, movespeed *= n, jumpvel *= n;
		return *this;
	}
};

enum
{
	EQUIP_LHAND = 0,
	EQUIP_RHAND,
	EQUIP_FEET,
	EQUIP_LEGS,
	EQUIP_TORSO,
	EQUIP_MISC1,
	EQUIP_MISC2,
	EQUIP_MAX
};

enum
{
	SLOT_LHAND = 1<<0,
	SLOT_RHAND = 1<<1,
	SLOT_FEET =  1<<2,
	SLOT_LEGS =  1<<3,
	SLOT_TORSO = 1<<4,
	SLOT_MISC1 = 1<<5,
	SLOT_MISC2 = 1<<6,
	SLOT_MAX =   1<<7
};

struct response
{
	const char *talk;
	int dest;
	const char *script;

	response(const char *t, int d, const char *s) : talk(t), dest(d), script(s)
	{}
	~response(){}
};

struct rpgchat
{
	const char *script; //when the dialogue is launched, this is executed to generate the instance, with any racial, ability or otherwise mutations of the dialogue
	const char *talk; //the words the object will present to the player

	vector<response *> dests;

	void close()
	{
		dests.setsize(0);
	}

	void open() { execute(script); }
	rpgchat(const char *s, const char *t) : script(s), talk(t) {}
	~rpgchat()
	{
		dests.deletecontentsp();
	};
};

struct rpgchar
{
	stats base, attributes;
	//attributes
	float health, mana;
	int lastpain, lastaction, lastdeath, experience, level, statpoints;
	int lastmana, lasthealth;
	bool updatestats, firstattack, secondattack;
	vector<projectile *> projs;
	rpgent *selected[EQUIP_MAX], *selectedspell;
	vector<rpgent *> inventory;
	vector<rpgent *> spellbook;

	rpgchar()
	{
		base = attributes = stats();
		health = base.maxhealth;
		mana = base.maxmana;
		statpoints = experience = 0;
		level = 1;
		loopi(EQUIP_MAX) selected[i] = NULL;
		selectedspell = NULL;
		lastmana = lasthealth = lastmillis;
		firstattack = secondattack = false;
		updatestats = true;
	}
	~rpgchar()
	{
		projs.deletecontentsp();
	}
};

//for damage, use status health, with a negative amount and a millis of 0.
//don't compress spell into item, it's a seperate type, they can't coexist within an entity with the current setup
enum
{
	ITYPE_ORNAMENT = 0,
	ITYPE_CONSUMABLE,
	ITYPE_QUEST,
	ITYPE_MELEE,
	ITYPE_RANGED,
	ITYPE_THROWN
};

struct rpgitem
{
	stats requirements, equipbonus, invbonus;
	int cooldown, range, type, slots;

	rpgitem()
	{
		requirements = equipbonus = invbonus = stats(true);
		cooldown = 250;
		range = 24;
		type = ITYPE_MELEE;
		slots = SLOT_RHAND|SLOT_LHAND;
	}
	~rpgitem() {}
};

enum
{
	STYPE_TARGET = 0,
	STYPE_CONE,
	STYPE_SELF
};

struct rpgspell
{
	stats requirements;
	int cooldown, range, type, cost, castdelay, effect, fade, gravity;

	rpgspell()
	{
		requirements = stats(true);
		cooldown = 500;
		range = 128;
		type = STYPE_TARGET;
		cost = 12;
		castdelay = 100;
		effect = 0;
		gravity = 200;
	}
	~rpgspell() {}
};

struct rpgobject
{
	bool active, triggered;
	int lastaction;

	rpgobject()
	{
		triggered = false;
		active = true;
		lastaction = 0;
	}
	~rpgobject() {}
};

enum
{
	ENT_CHAR = 0,
	ENT_ITEM,
	ENT_SPELL,
	ENT_OBJECT,
	ENT_MAX
};

//ent temporaries, things which update each frame, save cycles by storing the data for the frame
struct tempent
{
	int dist;
	float fade;

	tempent() : dist(0), fade(1) {}
	~tempent() {}
};

struct rpgent : dynent
{
	tempent temp;

	int etype, spawn; //spawn is used by respawnable monsters
	const char *model;
	const char *name, *description, *icon;

	vector<status> effects;

	int chatpos;
	vector<rpgchat *> dialogue; //note, we can script dialogues and other miscelaneous events onto the entity, such as portals with multiple destinations, or items which might have regrettable effects

	rpgchar *character;
	rpgitem *item;
	rpgspell *spell;
	rpgobject *object; //for interactable objects, switches portals, doors, etc

	const char *interact, //script to execute when interacted with
		*approach, //script to execute when another creature is within range
		*death; //script to execute upon death

	void cleansubtypes()
	{
		if(character)
		{
			character->inventory.deletecontentsp();
			character->spellbook.deletecontentsp();
			delete character; character = NULL;
		}
		if(item) {delete item; item = NULL;}
		if(spell) {delete spell; spell = NULL;}
		if(object) {delete object; object = NULL;}
	}

	rpgent(const char *m, int t = ENT_CHAR) : etype(t), spawn(-1), model(newstring(m)), chatpos(-1)
	{
		interact = approach = death = name = NULL;
		icon = newstring("empty");
		description = newstring("No description");
		character = NULL; item = NULL; spell = NULL; object = NULL;
		switch(etype)
		{
			case ENT_CHAR:
				character = new rpgchar(); break;
			case ENT_ITEM:
				item = new rpgitem(); break;
			case ENT_SPELL:
				spell = new rpgspell(); break;
			case ENT_OBJECT:
				object = new rpgobject(); break;
		}
	}
	~rpgent()
	{
		cleansubtypes();
		dialogue.deletecontentsp();
	}

	bool equip(rpgent *item, int slot = -1)
	{
		if(item->etype == ENT_ITEM)
		{
			slot = clamp(EQUIP_MAX-1, 0, slot);
			if(character->attributes < item->item->requirements) return false;
			if(item->item->slots&(1<<slot))
			{
				dequip(item); //just in case the item is in other slots
				character->selected[slot] = item;
				character->updatestats = true;
				return true;
			}
			return false;
		}
		else //if(item->etype==ENT_SPELL)
		{
			if(character->attributes < item->spell->requirements) return false;
			character->selectedspell = item;
			return true;
		}
	}

	bool dequip(rpgent *item)
	{
		loopi(EQUIP_MAX)
		{
			if(character->selected[i] == item)
			{
				character->selected[i] = NULL;
				return true;
			}
		}
		return false;
	}

	void respawn(bool death = false)
	{
		dynent::reset();
		switch(etype)
		{
			case ENT_CHAR:
				if(death)
				{
					character->health = 1;
					character->mana = 10;
					effects.add(status(STATUS_DEATH, 0, 20000));
				}
				else
				{
					character->health = character->attributes.maxhealth;
					character->mana = character->attributes.maxmana;
				}
				character->lastpain = character->lastaction = 0;
				character->lastdeath = character->lasthealth = character->lastmana = lastmillis;
				state = CS_ALIVE;
				break;
			case ENT_SPELL:
			case ENT_ITEM:
			case ENT_OBJECT:
			default:
				break;
		}
	}

	void die(int &lastmillis)
	{
		if(etype != ENT_CHAR) return;

		state = CS_DEAD;
		character->lastpain = character->lastaction = lastmillis;
		character->health = character->mana = 0;
		effects.setsize(0);

		if(death)
		{
			//TODO set aliases
			execute(death);
		}
	}

	void reset() //only used for the player
	{
		cleansubtypes();
		character = new rpgchar();
		effects.setsize(0);
	}

	void newmodel(bool init = false)
	{
		if(!init) o.z -= eyeheight;
		setbbfrommodel(this, model);
		if(!init) o.z += eyeheight;
	}

	void takedamage(float amount, int type)
	{
		if(!character) return;
		#define resist(n, r) if((type & n) == n) \
		{\
			amount *= (100 - character->attributes.resistance[r]) / 100.0f; \
			type -= n; \
		}

		resist(ATTACK_MAGIC, RESIST_MAGIC)
		else resist(ATTACK_PIERCE, RESIST_PIERCE)
		else resist(ATTACK_BLUNT, RESIST_BLUNT)
		else resist(ATTACK_SLASHING, RESIST_SLASH) //this will succeed always

		if(type)
		{
			amount *= 1.25;
			resist(ATTACK_EARTH, RESIST_EARTH)
			else resist(ATTACK_AIR, RESIST_AIR)
			else resist(ATTACK_WATER, RESIST_WATER)
			else resist(ATTACK_FIRE, RESIST_FIRE)
			else
			{
				conoutf("invalid resistance");
				amount *= .8;
			}
		}
		character->health -= amount;

		#undef resist
	}

	float geteffectmul(rpgent &wep)
	{
		switch(wep.etype)
		{
			case ENT_ITEM:
				return 1.0f;
			case ENT_SPELL:
				return (character->attributes.intelligence / 2.0f + character->attributes.charisma / 4.0f + character->attributes.skills[SKILL_MAGIC].level) / 50.0f; //max is 3.5 default would be .5
			default:
				return 1.0f;
		}
	}
};

/* struct rpgai : rpgent
{
	int state,
		friendliness;
}; */

enum
{
	S_JUMP = 0, S_LAND, S_RIFLE, S_TELEPORT, S_SPLASH1, S_SPLASH2, S_CG,
	S_RLFIRE, S_RUMBLE, S_JUMPPAD, S_WEAPLOAD, S_ITEMAMMO, S_ITEMHEALTH,
	S_ITEMARMOUR, S_ITEMPUP, S_ITEMSPAWN,  S_NOAMMO, S_PUPOUT,
	S_PAIN1, S_PAIN2, S_PAIN3, S_PAIN4, S_PAIN5, S_PAIN6,
	S_DIE1, S_DIE2,
	S_FLAUNCH, S_FEXPLODE,
	S_SG, S_PUNCH1,
	S_GRUNT1, S_GRUNT2, S_RLHIT,
	S_PAINO,
	S_PAINR, S_DEATHR,
	S_PAINE, S_DEATHE,
	S_PAINS, S_DEATHS,
	S_PAINB, S_DEATHB,
	S_PAINP, S_PIGGR2,
	S_PAINH, S_DEATHH,
	S_PAIND, S_DEATHD,
	S_PIGR1, S_ICEBALL, S_SLIMEBALL, S_PISTOL,

	S_V_BASECAP, S_V_BASELOST,
	S_V_FIGHT,
	S_V_BOOST, S_V_BOOST10,
	S_V_QUAD, S_V_QUAD10,
	S_V_RESPAWNPOINT,

	S_FLAGPICKUP,
	S_FLAGDROP,
	S_FLAGRETURN,
	S_FLAGSCORE,
	S_FLAGRESET,

	S_BURN,
	S_CHAINSAW_ATTACK,
	S_CHAINSAW_IDLE,

	S_HIT
};

namespace entities
{
	extern vector<rpgentity *> ents;
	extern void spawnfroment(int i);
	extern void startmap();
	extern void teleport(rpgent &d, int dest);
	extern void renderentities(bool mainpass);
};

namespace game
{
	extern bool transfer;
	extern rpgent *player1, *selected, *lastcreated;
	extern vector<rpgent *> rpgobjs;
	extern int rpgobjdist;
	extern dynent *intersectclosest(const vec &from, const vec &to, rpgent *at, float maxdist = 1e16f);
	extern bool intersect(dynent *d, const vec &from, const vec &to);
	extern const char *geteffecticon(status &s);
	extern const char *geteffectdescription(status &s);
};

namespace rpgobj
{
	extern vector<projeffect *> effects;
	extern void pickup(rpgent *d, rpgent *o);
	extern int getident(rpgent *d);
	extern int rpgobjupdatedist;
	extern void update();
	extern void adddynlights();
	extern void dropitem(rpgent *item, rpgent *parent);
};

namespace gui
{
	extern bool guisopen();
	extern void render();
	extern void clearguis();
}

#endif
